
///////////////////////////////////////////////////////////////////////////////
//
//   ROCKWELL SEMICONDUCTOR SYSTEMS - COMMERCIAL GPS BUSINESS
//
///////////////////////////////////////////////////////////////////////////////
//
//
//   GPSCOM.C - GPS Receiver Serial Communications
// 
//
//   DESCRIPTION
//
//   Functions to support the changing of the receiver's serial communication
//   parameters (e.g., baud, parity, etc.).
//
//
//   REVISION HISTORY
//
//   $Log:   V:\Projects\Labmon\Source\Archives\GPSCOM.C_V  $
//   
//      Rev 1.15   09 Nov 1998 10:44:14   phungh
//   labmon51: Fix file logging and missing
//   logged data bugs.  Changes made to the 
//   "Pause" variable ( 1 -> 0 ) to allow main
//   loop continue.  Move the "write to file"
//   code portion out of the interrupt handler
//   so that data is not missed because of time
//   spent too long in the interrupt handler
//   
//      Rev 1.3   Jul 09 1997 10:34:12   COLEJ
//    
//   
//      Rev 1.2   Feb 12 1997 16:05:00   COLEJ
//    
//   
//      Rev 1.1   Aug 13 1996 13:13:48   COLEJ
//    
//   
//      Rev 1.0   13 May 1996 14:52:30   GPSADMIN
//   Initial release to version control.
//
//
////////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <graph.h>

#include "gencons.h"
#include "gentypes.h"

#include "labmon.h"
#include "initfunc.h"
#include "monmisc.h"
#include "sendnmea.h"
#include "ztypes.h"

extern tDATATYPE DataType;

extern tULONG cnt217;
extern tULONG cnticom;
extern tULONG Cnt1330;

extern unsigned char tx_buff[MAX_TX_BUFFER];


// NSetSerialComm - sets serial port communication parameters for NavCore
//                  series of receivers to user-supplied values.
//
tVOID NSetSerialComm ( tVOID )
{
   tLONG   auxbaud = 9600;
   tCHAR   auxprty = 'N';
   tSHORT    auxdbit = 8;
   tSHORT    auxsbit = 1;
   tSHORT    auxbaudnum, auxprtynum;
   tCHAR   nmea_msg[83];
   tSHORT    valid;

   clear_message_line();
               
   // Prompt for changes to receiver serial port...

   // Get aux port settings from user.
   //
   auxbaud = ask_long("AUX PORT BAUD RATE (3/6/12/24/48/9600) -> ", auxbaud, &valid); 
   clear_message_line(); 
   if(!valid) return;

   auxprty = ask_char("AUX PORT PARITY (N,O,E) -> ", auxprty, &valid); 
   auxprty = toupper(auxprty);                     
   clear_message_line();
   if(!valid) return;

   auxdbit = ask_short("AUX PORT DATA BITS (7,8) -> ", auxdbit, &valid);
   clear_message_line();
   if(!valid) return;

   auxsbit = ask_short("AUX PORT STOP BITS (1,2) -> ", auxsbit, &valid);
   clear_message_line(); 
   if(!valid) return;

   // Check aux port settings.
   //
   switch(auxbaud){
      case    300: auxbaudnum = 0 ; break;
      case    600: auxbaudnum = 1 ; break;
      case   1200: auxbaudnum = 2 ; break;
      case   2400: auxbaudnum = 3 ; break;
      case   4800: auxbaudnum = 4 ; break;
      case   9600: auxbaudnum = 5 ; break;
      default:     valid = 0;
   }
   switch(toupper(auxprty)){
      case 78: auxprtynum = 2; break;     // none
      case 79: auxprtynum = 0; break;     // odd
      case 69: auxprtynum = 1; break;     // even
      default: valid = 0; 
   }

   if(!valid){
      clear_message_line();  
      _outtext("INVALID PORT SETTINGS"); 
      return;
   }                 
      
   // Build port configuration message.
   //       
   if(DataType == NC_NMEA){
      sprintf(nmea_msg,
         "$PRWIICOM,,,%i,%c,%i,%i,",auxbaud,auxprty,auxsbit,auxdbit);
      cnticom++;
      send_nmea_msg(nmea_msg);
      wait(1.5f);
   }       
   else if(DataType == NC_BINARY){
      make_tx_header(217, 13);
      *(unsigned short *)(tx_buff+14) = auxbaudnum;
      *(unsigned short *)(tx_buff+16) = auxprtynum;
      *(unsigned short *)(tx_buff+18) = auxsbit;
      *(unsigned short *)(tx_buff+20) = auxdbit;
      cnt217++; 
      send_msg(tx_buff);
   }
}


// ZSetSerialComm - sets serial port communication parameters for Zodiac
//                  series of receivers to user-supplied values.
//
tVOID ZSetSerialComm ( tVOID )
{
   tLONG   hostbaud = 9600, auxbaud = 9600;
   tCHAR   hostprty = 'N',  auxprty = 'N';
   tSHORT  hostdbit = 8,    auxdbit = 8;
   tSHORT  hostsbit = 1,    auxsbit = 1;
   tSHORT  hostpre  = 0,    auxpre  = 0;
   tSHORT  hostpost = 0,    auxpost = 0;
   tSHORT  hostbaudnum, auxbaudnum;
   tSHORT  hostprtynum, auxprtynum;
   tMSGBUF ZMsgBuf;
   tCHAR   ctemp;
   tSHORT  valid;
   tUSHORT len;
   
   // clear port validity flags
   SPValidData.Port1 = 0;
   SPValidData.Port2 = 0;
   
   clear_message_line();
   memset(&ZMsg.Msg1330, 0, sizeof(ZMsg.Msg1330));
               
   // Prompt for changes to receiver serial ports...
           
   // Get host port settings from user.
   //
   ctemp = ask_char("MODIFY HOST PORT SETTINGS? (Y,N) -> ",'N',&valid);
   clear_message_line();  
   if(!valid) return;

   if(toupper(ctemp) == 'Y'){
      hostbaud = ask_long(
         "HOST PORT BAUD RATE (0/3/6/12/24/48/96/192/384/576/768/115200) -> ",
         hostbaud, &valid);
      clear_message_line();
      if(!valid) return;

      hostprty = ask_char("HOST PORT PARITY (N,O,E) -> ", hostprty, &valid);
      hostprty = toupper(hostprty);
      clear_message_line();
      if(!valid) return;

      hostdbit = ask_short("HOST PORT DATA BITS (7,8) -> ", hostdbit, &valid);
      clear_message_line();
      if(!valid) return;

      hostsbit = ask_short("HOST PORT STOP BITS (1,2) -> ", hostsbit, &valid);
      clear_message_line();
      if(!valid) return;

      // Get custom rate.
      //
      if(hostbaud == 0){
         clear_message_line();
         hostpre = ask_short("HOST PORT PRE-SCALE (0-255) -> ", hostpre, &valid);
         clear_message_line();
         if(!valid) return;

         hostpost = ask_short("HOST POST POST-SCALE (0-7) -> ", hostpost, &valid);
         clear_message_line();
         if(!valid) return;
      }
      SPValidData.Port1 = 1;
   }

   // Get aux port settings from user.
   //
   ctemp = ask_char("MODIFY AUX PORT SETTINGS? (Y,N) -> ",'N',&valid);
   clear_message_line();  
   if(!valid) return;
              
   if(toupper(ctemp) == 'Y'){
      auxbaud = ask_long(
         "AUX PORT BAUD RATE (0/3/6/12/24/48/96/192/384/576/768/115200) -> ",
         auxbaud, &valid); 
      clear_message_line(); 
      if(!valid) return;

      auxprty = ask_char("AUX PORT PARITY (N,O,E) -> ", auxprty, &valid);
      auxprty = toupper(auxprty);        
      clear_message_line();
      if(!valid) return;

      auxdbit = ask_short("AUX PORT DATA BITS (7,8) -> ", auxdbit, &valid); 
      clear_message_line();
      if(!valid) return;

      auxsbit = ask_short("AUX PORT STOP BITS (1,2) -> ", auxsbit, &valid);
      clear_message_line(); 
      if(!valid) return;

      // Get custom rate.
      //
      if(auxbaud == 0){
         clear_message_line();
         auxpre = ask_short("AUX PORT PRE-SCALE (0-255) -> ", auxpre, &valid);
         clear_message_line(); 
         if(!valid) return;

         auxpost = ask_short("AUX POST POST-SCALE (0-7) -> ", auxpost, &valid);
         clear_message_line();
         if(!valid) return;
      }
      SPValidData.Port2 = 2;
   }
      
   // Check host port settings.
   //
   switch(hostbaud){  
      case      0: hostbaudnum = 0 ; break;
      case    300: hostbaudnum = 1 ; break;
      case    600: hostbaudnum = 2 ; break;
      case   1200: hostbaudnum = 3 ; break;
      case   2400: hostbaudnum = 4 ; break;
      case   4800: hostbaudnum = 5 ; break;
      case   9600: hostbaudnum = 6 ; break;      
      case  19200: hostbaudnum = 7 ; break;      
      case  38400: hostbaudnum = 8 ; break;      
      case  57600: hostbaudnum = 9 ; break;      
      case  76800: hostbaudnum = 10; break;      
      case 115200: hostbaudnum = 11; break;    
      default:     valid = 0;
   }
   switch(toupper(hostprty)){
      case 78: hostprtynum = 0; break;   // none
      case 79: hostprtynum = 1; break;   // odd
      case 69: hostprtynum = 2; break;   // even    
      default: valid = 0;
   }

   // Check aux port settings.
   //
   switch(auxbaud){  
      case      0: auxbaudnum = 0 ; break;
      case    300: auxbaudnum = 1 ; break;
      case    600: auxbaudnum = 2 ; break;
      case   1200: auxbaudnum = 3 ; break;
      case   2400: auxbaudnum = 4 ; break;
      case   4800: auxbaudnum = 5 ; break;
      case   9600: auxbaudnum = 6 ; break;          
      case  19200: auxbaudnum = 7 ; break;          
      case  38400: auxbaudnum = 8 ; break;          
      case  57600: auxbaudnum = 9 ; break;          
      case  76800: auxbaudnum = 10; break;          
      case 115200: auxbaudnum = 11; break;        
      default:     valid = 0;
   }
   switch(toupper(auxprty)){
      case 78: auxprtynum = 0; break;     // none
      case 79: auxprtynum = 1; break;     // odd
      case 69: auxprtynum = 2; break;     // even
      default: valid = 0; 
   }

   // Notify if settings are invalid.
   //
   if(!valid){
      clear_message_line();  
      _outtext("INVALID PORT SETTINGS"); 
      return;
   }

   // Build port configuration message.
   //
   ZMsg.Msg1330.Validity = SPValidData.Port1 | 
                           SPValidData.Port2;
                                 
   if(SPValidData.Port1){
      ZMsg.Msg1330.Port1.DataBits  = (tUSHORT) hostdbit - 7;
      ZMsg.Msg1330.Port1.StopBits  = (tUSHORT) hostsbit - 1;
      ZMsg.Msg1330.Port1.Parity    = (tUSHORT) hostprtynum;
      ZMsg.Msg1330.Port1.BitRate   = (tUSHORT) hostbaudnum;
      ZMsg.Msg1330.Port1.Prescale  = (tUSHORT) hostpre;
      ZMsg.Msg1330.Port1.Postscale = (tUSHORT) hostpost;
   }
   if(SPValidData.Port2){
      ZMsg.Msg1330.Port2.DataBits  = (tUSHORT) auxdbit - 7;  
      ZMsg.Msg1330.Port2.StopBits  = (tUSHORT) auxsbit - 1;
      ZMsg.Msg1330.Port2.Parity    = (tUSHORT) auxprtynum;
      ZMsg.Msg1330.Port2.BitRate   = (tUSHORT) auxbaudnum;
      ZMsg.Msg1330.Port2.Prescale  = (tUSHORT) auxpre;
      ZMsg.Msg1330.Port2.Postscale = (tUSHORT) auxpost;
   }                     
   if(ZMsg.Msg1330.Validity){
      BldZMsg(&ZMsgBuf, ZMsg, 1330);
      len = sizeof(ZMsg.Msg1330) + HDR_LEN;
      memcpy(tx_buff, &ZMsgBuf, len);
      Cnt1330++; 
      send_msg(tx_buff);       
   }
   else{
      clear_message_line();  
      _outtext("NEITHER PORT SETTINGS VALID");  
   }
}
